<?php
/**
 * Plugin Name: DCW Broken Link Checker Lite
 * Description: Scan a small set of pages for broken links (404/timeout). Manual scan and optional daily scan with email alerts.
 * Version: 1.0.2
 * Author: DCWireNet
 * License: GPL2
 * Text Domain: dcw-broken-link-checker-lite
 */

if ( ! defined('ABSPATH') ) exit;

define('DCW_BLCL_VERSION', '1.0.2');
define('DCW_BLCL_DIR', plugin_dir_path(__FILE__));
define('DCW_BLCL_URL', plugin_dir_url(__FILE__));

// Safe require helper
function dcw_blcl_safe_require( $rel ) {
    $path = DCW_BLCL_DIR . ltrim($rel, '/');
    if ( file_exists($path) ) { require_once $path; return true; }
    add_action('admin_notices', function() use ($rel,$path){
        if ( current_user_can('activate_plugins') ) {
            echo '<div class="notice notice-error"><p><strong>DCW Broken Link Checker Lite:</strong> Missing file <code>' . esc_html($rel) . '</code> (looked at <code>' . esc_html($path) . '</code>).</p></div>';
        }
    });
    return false;
}

// Load classes
$ok_admin   = dcw_blcl_safe_require('includes/class-dcw-blcl-admin.php');
$ok_scanner = dcw_blcl_safe_require('includes/class-dcw-blcl-scanner.php');

if ( $ok_admin && $ok_scanner ) {
    add_action('plugins_loaded', function () {
        DCW_BLCL_Admin::init();
    });
}

// === Cron wiring ===
// Custom hook fired by WP-Cron
add_action('dcw_blcl_cron_scan', function() {
    if ( ! class_exists('DCW_BLCL_Admin') || ! class_exists('DCW_BLCL_Scanner') ) return;
    DCW_BLCL_Admin::cron_scan(); // run scan + save + maybe email
});

// Schedule/unschedule on (de)activation
register_activation_hook(__FILE__, function () {
    // If daily_scan is already on in saved settings, schedule the event.
    $opts = get_option('dcw_blcl_settings', []);
    $daily = !empty($opts['daily_scan']);
    if ( $daily && ! wp_next_scheduled('dcw_blcl_cron_scan') ) {
        // schedule to run daily at next 2:17am local (arbitrary, stable)
        $ts = strtotime('tomorrow 02:17');
        if ( ! $ts ) $ts = time() + DAY_IN_SECONDS;
        wp_schedule_event($ts, 'daily', 'dcw_blcl_cron_scan');
    }
});

register_deactivation_hook(__FILE__, function () {
    $t = wp_next_scheduled('dcw_blcl_cron_scan');
    if ( $t ) wp_unschedule_event($t, 'dcw_blcl_cron_scan');
});
