<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_BLCL_Admin {

    public static function init() {
        add_action('admin_menu', [__CLASS__, 'menu']);
        add_action('admin_init', [__CLASS__, 'register_settings']);

        // Manual scan handler
        add_action('admin_post_dcw_blcl_scan_now', [__CLASS__, 'handle_scan']);
    }

    public static function register_settings() {
        register_setting('dcw_blcl_group', 'dcw_blcl_settings', [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
            'default'           => [],
        ]);
    }

    public static function sanitize($in) {
        $in   = is_array($in) ? $in : [];
        $prev = get_option('dcw_blcl_settings', []);

        $out = [];
        $out['scope']         = isset($in['scope']) && in_array($in['scope'], ['home','home_plus_latest'], true) ? $in['scope'] : 'home';
        $out['limit']         = isset($in['limit']) ? max(1, (int)$in['limit']) : 150;
        $out['timeout']       = isset($in['timeout']) ? max(1, (int)$in['timeout']) : 8;
        $out['internal_only'] = !empty($in['internal_only']) ? 1 : 0;

        // Email alerts
        $out['email_alerts']  = !empty($in['email_alerts']) ? 1 : 0;
        $out['email_to']      = isset($in['email_to']) ? sanitize_text_field($in['email_to']) : '';

        // Daily scan toggle
        $out['daily_scan']    = !empty($in['daily_scan']) ? 1 : 0;

        // (Un)schedule cron if toggle changed
        $was_daily = ! empty($prev['daily_scan']);
        if ( $was_daily !== (bool)$out['daily_scan'] ) {
            if ( $out['daily_scan'] ) {
                if ( ! wp_next_scheduled('dcw_blcl_cron_scan') ) {
                    $ts = strtotime('tomorrow 02:17');
                    if ( ! $ts ) $ts = time() + DAY_IN_SECONDS;
                    wp_schedule_event($ts, 'daily', 'dcw_blcl_cron_scan');
                }
            } else {
                $t = wp_next_scheduled('dcw_blcl_cron_scan');
                if ( $t ) wp_unschedule_event($t, 'dcw_blcl_cron_scan');
            }
        }

        return $out;
    }

    public static function menu() {
        add_management_page(
            'DCW Broken Link Checker Lite',
            'Broken Link Checker (Lite)',
            'manage_options',
            'dcw-blcl',
            [__CLASS__, 'render_page']
        );
    }

    /**
     * Send an email summary for a scan (manual or cron).
     * Always sends when email alerts are enabled, even with 0 broken links.
     */
  private static function send_scan_email( array $payload, array $opts, string $context ) : void {
        if ( empty($opts['email_alerts']) ) return;

        $to = sanitize_text_field( $opts['email_to'] ?: get_option('admin_email') );
        if ( ! is_email($to) ) return;

        $site_name = wp_specialchars_decode( get_bloginfo('name'), ENT_QUOTES );

        // Make a human-readable, non-clickable site URL to avoid SafeLinks wrapping.
        $site_url       = home_url('/');
        $site_url_text  = preg_replace('#^(https?)://#i', '$1[: ]//', $site_url); // https[:]//example.com/
        // Fallback host text (also not a link)
        $site_host      = wp_parse_url($site_url, PHP_URL_HOST);

        $when  = isset($payload['ts']) && $payload['ts'] ? date_i18n( get_option('date_format') . ' ' . get_option('time_format'), (int)$payload['ts'] ) : current_time('mysql');
        $scope = isset($opts['scope']) ? $opts['scope'] : 'home';

        $checked = (int)($payload['total_links'] ?? 0);
        $broken  = is_array($payload['broken'] ?? null) ? count($payload['broken']) : 0;
        $pages   = (int)($payload['pages_scanned'] ?? 0);
        $dur     = isset($payload['duration']) ? (float)$payload['duration'] : 0.0;

        $subject = sprintf('[%s] Broken Link Checker: %d broken, %d checked', $site_name, $broken, $checked);

        $lines = [];
        $lines[] = sprintf('Site: %s', $site_url_text);
        $lines[] = sprintf('When: %s', $when);
        $lines[] = sprintf('Context: %s', $context);
        $lines[] = sprintf('Scope: %s', $scope);
        $lines[] = sprintf('Pages scanned: %d', $pages);
        $lines[] = sprintf('Links checked: %d', $checked);
        $lines[] = sprintf('Broken links: %d', $broken);
        if ( $dur > 0 ) {
            $lines[] = sprintf('Duration: ~%.2fs', $dur);
        }
        $lines[] = '';
        $lines[] = 'Dashboard path: Tools > Broken Link Checker (Lite)';
        $lines[] = '';

        if ( $broken > 0 ) {
            $lines[] = 'Broken link details:';
            foreach ( (array)$payload['broken'] as $b ) {
                $url    = isset($b['url']) ? (string)$b['url'] : '';
                $code   = isset($b['code']) ? (int)$b['code'] : 0;
                $reason = isset($b['reason']) ? (string)$b['reason'] : '';
                $lines[] = sprintf('- %s  (%d %s)', $url, $code, $reason);
            }
        } else {
            $lines[] = 'No broken links were found in this scan.';
        }

        $headers = ['Content-Type: text/plain; charset=UTF-8'];
        wp_mail($to, $subject, implode("\n", $lines), $headers);
    }

    /** Shared save logic for manual and cron scans (now always emails if enabled). */
    private static function save_results_and_maybe_email( $result, $context = 'manual' ) {
        $now   = time();
        $opts  = get_option('dcw_blcl_settings', []);
        $scope = isset($opts['scope']) ? $opts['scope'] : 'home';

        $pages_scanned = ($scope === 'home_plus_latest') ? 11 : 1;

        $payload = [
            'ts'            => $now,
            'duration'      => isset($result['duration']) ? (float)$result['duration'] : 0.0,
            'total_links'   => isset($result['checked'])  ? (int)$result['checked']  : 0,
            'pages_scanned' => $pages_scanned,
            'broken'        => isset($result['broken']) && is_array($result['broken']) ? array_values($result['broken']) : [],
            'context'       => $context,
        ];

        update_option('dcw_blcl_last', $payload, false);

        // ALWAYS email when alerts are enabled (manual & cron), even if 0 broken.
        self::send_scan_email($payload, $opts, $context);
    }

    /** Manual scan (admin button). */
    public static function handle_scan() {
        if ( ! current_user_can('manage_options') ) {
            wp_die(__('Sorry, you are not allowed to do that.', 'dcw-blcl'));
        }
        check_admin_referer('dcw_blcl_scan');

        if ( ! class_exists('DCW_BLCL_Scanner') ) {
            wp_die('Scanner class missing.');
        }

        $result = DCW_BLCL_Scanner::scan_now();
        self::save_results_and_maybe_email($result, 'manual');

        wp_redirect( add_query_arg('dcw_blcl_scanned', '1', admin_url('tools.php?page=dcw-blcl')) );
        exit;
    }

    /** Cron scan (scheduled). */
    public static function cron_scan() {
        if ( ! class_exists('DCW_BLCL_Scanner') ) return;
        $result = DCW_BLCL_Scanner::scan_now();
        self::save_results_and_maybe_email($result, 'cron');
    }

    private static function get_last_results() {
        $last = get_option('dcw_blcl_last', []);
        if ( ! is_array($last) ) $last = [];
        return [
            'ts'            => isset($last['ts']) ? (int)$last['ts'] : 0,
            'duration'      => isset($last['duration']) ? (float)$last['duration'] : 0.0,
            'total_links'   => isset($last['total_links']) ? (int)$last['total_links'] : 0,
            'pages_scanned' => isset($last['pages_scanned']) ? (int)$last['pages_scanned'] : 0,
            'broken'        => isset($last['broken']) && is_array($last['broken']) ? $last['broken'] : [],
            'context'       => isset($last['context']) ? $last['context'] : '',
        ];
    }

    public static function render_page() {
        if ( ! current_user_can('manage_options') ) return;

        $opts = get_option('dcw_blcl_settings', []);
        $opts = wp_parse_args($opts, [
            'scope'         => 'home',
            'limit'         => 150,
            'timeout'       => 8,
            'internal_only' => 0,
            'email_alerts'  => 0,
            'email_to'      => '',
            'daily_scan'    => 0,
        ]);

        $last = self::get_last_results();
        ?>
        <div class="wrap">
            <h1>DCW Broken Link Checker Lite</h1>

            <?php if ( isset($_GET['dcw_blcl_scanned']) ): ?>
                <div class="notice notice-success is-dismissible"><p>Scan completed.</p></div>
            <?php endif; ?>

            <h2 class="title">Settings</h2>
            <form method="post" action="options.php">
                <?php settings_fields('dcw_blcl_group'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="dcw_blcl_scope">Scope</label></th>
                        <td>
                            <select name="dcw_blcl_settings[scope]" id="dcw_blcl_scope">
                                <option value="home" <?php selected('home', $opts['scope']); ?>>Home page only</option>
                                <option value="home_plus_latest" <?php selected('home_plus_latest', $opts['scope']); ?>>Home + latest 10 posts/pages</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="dcw_blcl_limit">Max links to check</label></th>
                        <td><input type="number" min="1" name="dcw_blcl_settings[limit]" id="dcw_blcl_limit" value="<?php echo esc_attr((int)$opts['limit']); ?>" class="small-text"> links</td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="dcw_blcl_timeout">HTTP timeout (seconds)</label></th>
                        <td><input type="number" min="1" name="dcw_blcl_settings[timeout]" id="dcw_blcl_timeout" value="<?php echo esc_attr((int)$opts['timeout']); ?>" class="small-text"></td>
                    </tr>
                    <tr>
                        <th scope="row">Internal links only</th>
                        <td><label><input type="checkbox" name="dcw_blcl_settings[internal_only]" value="1" <?php checked(1, (int)$opts['internal_only']); ?>> Only check links on this domain</label></td>
                    </tr>
                    <tr>
                        <th scope="row">Email alerts</th>
                        <td>
                            <label><input type="checkbox" name="dcw_blcl_settings[email_alerts]" value="1" <?php checked(1, (int)$opts['email_alerts']); ?>> Send a summary email after every scan</label><br>
                            <input type="email" class="regular-text" placeholder="<?php echo esc_attr( get_option('admin_email') ); ?>" name="dcw_blcl_settings[email_to]" value="<?php echo esc_attr($opts['email_to']); ?>">
                            <p class="description">Leave blank to send to the site admin email.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Daily scan (WP-Cron)</th>
                        <td>
                            <label><input type="checkbox" name="dcw_blcl_settings[daily_scan]" value="1" <?php checked(1, (int)$opts['daily_scan']); ?>> Run once per day (emails the summary too)</label>
                            <p class="description">WP-Cron runs on site traffic. For accurate timing, configure a real server cron to call <code>wp-cron.php</code>.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Save Settings'); ?>
            </form>

            <hr>

            <h2 class="title">Manual Scan</h2>
            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                <?php wp_nonce_field('dcw_blcl_scan'); ?>
                <input type="hidden" name="action" value="dcw_blcl_scan_now">
                <?php submit_button('Scan Now', 'primary'); ?>
            </form>

            <hr>

            <h2 class="title">Last results</h2>
            <p>
                <strong>When:</strong>
                <?php
                    if ($last['ts'] > 0) {
                        echo esc_html( date_i18n( get_option('date_format') . ' ' . get_option('time_format'), $last['ts'] ) );
                        echo ' — ';
                    } else {
                        echo '— — ';
                    }
                ?>
                <strong>Pages:</strong> <?php echo esc_html( (string) $last['pages_scanned'] ); ?>
                &nbsp;&nbsp;
                <strong>Links:</strong> <?php echo esc_html( (string) $last['total_links'] ); ?>
                &nbsp;&nbsp;
                <strong>Broken:</strong> <?php echo esc_html( (string) count($last['broken']) ); ?>
                <?php if ($last['duration'] > 0): ?>
                    &nbsp;&nbsp;<em>(~<?php echo esc_html( number_format_i18n($last['duration'], 2) ); ?>s)</em>
                <?php endif; ?>
                <?php if ( ! empty($last['context']) ): ?>
                    &nbsp;&nbsp;<code><?php echo esc_html($last['context']); ?></code>
                <?php endif; ?>
            </p>

            <?php if ( ! empty($last['broken']) ): ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>URL</th>
                            <th>Status</th>
                            <th>Reason</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($last['broken'] as $row):
                        $url    = isset($row['url']) ? $row['url'] : '';
                        $code   = isset($row['code']) ? (int)$row['code'] : 0;
                        $reason = isset($row['reason']) ? $row['reason'] : '';
                    ?>
                        <tr>
                            <td><a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($url); ?></a></td>
                            <td><?php echo esc_html( (string)$code ); ?></td>
                            <td><?php echo esc_html( (string)$reason ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <p>No broken links recorded in the last scan.</p>
            <?php endif; ?>
        </div>
        <?php
    }
}
