<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_CBL_Admin {

    /** @var string */
    private static $screen_id = '';

    public static function init(){
        add_action('admin_menu', [__CLASS__, 'menu']);
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'assets']);
    }

    public static function menu(){
        $hook = add_options_page(
            __('DCW Country Blocker Lite', 'dcw-country-blocker-lite'),
            __('DCW Country Blocker Lite', 'dcw-country-blocker-lite'),
            'manage_options',
            'dcw-cbl',
            [__CLASS__, 'settings_page']
        );
        self::$screen_id = $hook;
    }

    public static function register_settings(){
        register_setting('dcw_cbl_group', 'dcw_cbl_settings', [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
            'default'           => []
        ]);
    }

    public static function assets($hook){
        if ( $hook !== self::$screen_id ) return;
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        // lightweight inline CSS for layout
        $css = '
        .dcw-cbl-field th{width:260px}
        .dcw-cbl-actions .button{margin-right:8px}
        .dcw-cbl-code{display:inline-block;padding:2px 6px;background:#f6f7f7;border:1px solid #dcdcde;border-radius:3px}
        ';
        wp_register_style('dcw-cbl-admin-inline', false);
        wp_enqueue_style('dcw-cbl-admin-inline');
        wp_add_inline_style('dcw-cbl-admin-inline', $css);
    }

    public static function sanitize($in){
        if ( ! is_array($in) ) $in = [];
        $out = [];

        $out['enabled']      = ! empty($in['enabled']) ? 1 : 0;

        // scope
        $scope = isset($in['scope']) ? (string) $in['scope'] : 'admin_login';
        $out['scope'] = in_array($scope, ['login','admin_login','site'], true) ? $scope : 'admin_login';

        // mode & countries
        $mode = isset($in['mode']) ? (string) $in['mode'] : 'block';
        $out['mode'] = in_array($mode, ['block','allow'], true) ? $mode : 'block';
        $countries = isset($in['countries']) ? strtoupper( (string) $in['countries'] ) : '';
        // normalize country list (AA, BB, CC)
        $list = array_filter(array_map('trim', explode(',', $countries)));
        $list = array_map(function($c){ return substr(preg_replace('/[^A-Z]/', '', $c), 0, 2); }, $list);
        $list = array_filter($list, function($c){ return (bool) preg_match('/^[A-Z]{2}$/', $c); });
        $out['countries'] = implode(', ', array_unique($list));

        // bypass & IP allow
        $out['admin_bypass'] = ! empty($in['admin_bypass']) ? 1 : 0;
        $out['ip_allow']     = isset($in['ip_allow']) ? sanitize_text_field( (string) $in['ip_allow'] ) : '';

        // action
        $action = isset($in['action']) ? (string) $in['action'] : 'blockpage';
        $out['action'] = in_array($action, ['blockpage','redirect'], true) ? $action : 'blockpage';
        $out['redirect_url'] = isset($in['redirect_url']) ? esc_url_raw( (string) $in['redirect_url'] ) : '';

        // block page styles
        $out['title']        = isset($in['title']) ? sanitize_text_field( (string) $in['title'] ) : 'Access Restricted';
        $out['message']      = isset($in['message']) ? wp_kses_post( (string) $in['message'] ) : '<p>Sorry, our site is not available in your region.</p>';
        $out['bg_color']     = isset($in['bg_color']) ? ( sanitize_hex_color( (string) $in['bg_color'] ) ?: '#0f172a' ) : '#0f172a';
        $out['bg_image']     = isset($in['bg_image']) ? esc_url_raw( (string) $in['bg_image'] ) : '';
        $out['bg_size']      = ( isset($in['bg_size']) && in_array($in['bg_size'], ['cover','contain'], true) ) ? (string) $in['bg_size'] : 'cover';
        $pos_allowed = ['top left','top center','top right','center left','center center','center right','bottom left','bottom center','bottom right'];
        $pos_in = isset($in['bg_position']) ? (string) $in['bg_position'] : 'center center';
        $out['bg_position']  = in_array($pos_in, $pos_allowed, true) ? $pos_in : 'center center';
        $out['title_color']  = isset($in['title_color']) ? ( sanitize_hex_color( (string) $in['title_color'] ) ?: '#ffffff' ) : '#ffffff';
        $out['text_color']   = isset($in['text_color'])  ? ( sanitize_hex_color( (string) $in['text_color'] )  ?: '#e5e7eb' ) : '#e5e7eb';

        // preview bypass (keep if you need)
        $out['preview_key']       = isset($in['preview_key']) ? sanitize_text_field( (string) $in['preview_key'] ) : '';
        $out['preview_allow_one'] = ! empty($in['preview_allow_one']) ? 1 : 0;

        // external lookup
        $out['ipinfo_token']      = isset($in['ipinfo_token']) ? sanitize_text_field( (string) $in['ipinfo_token'] ) : '';
        $out['external_timeout']  = isset($in['external_timeout']) ? max(1, (int) $in['external_timeout']) : 3;
        $policy = isset($in['api_failure_action']) ? (string) $in['api_failure_action'] : 'allow';
        $out['api_failure_action']= in_array($policy, ['allow','block'], true) ? $policy : 'allow';

        return $out;
    }

    public static function settings_page(){
        if ( ! current_user_can('manage_options') ) return;

        $o = get_option('dcw_cbl_settings', []);
        $o = wp_parse_args($o, [
            'enabled'=>0, 'scope'=>'admin_login', 'mode'=>'block', 'countries'=>'',
            'admin_bypass'=>1, 'ip_allow'=>'',
            'action'=>'blockpage', 'redirect_url'=>'',
            'title'=>'Access Restricted', 'message'=>'<p>Sorry, our site is not available in your region.</p>',
            'bg_color'=>'#0f172a', 'bg_image'=>'', 'bg_size'=>'cover', 'bg_position'=>'center center',
            'title_color'=>'#ffffff', 'text_color'=>'#e5e7eb',
            'preview_key'=>'', 'preview_allow_one'=>0,
            'ipinfo_token'=>'', 'external_timeout'=>3, 'api_failure_action'=>'allow',
        ]);

        // Build Preview (block page) URL – admin-only, nonce-protected
        $preview_url = add_query_arg(
            [
                'dcw_cbl_preview' => '1',
                '_dcw_cbl_nonce'  => wp_create_nonce('dcw_cbl_preview'),
            ],
            home_url('/')
        );

        // Build Diagnostics URL – admin-only, nonce-protected
        $diag_url = add_query_arg(
            [
                'dcw_cbl_diag'   => '1',
                '_dcw_cbl_nonce' => wp_create_nonce('dcw_cbl_diag'),
            ],
            home_url('/')
        );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('DCW Country Blocker Lite', 'dcw-country-blocker-lite'); ?></h1>

            <form method="post" action="options.php">
                <?php settings_fields('dcw_cbl_group'); ?>

                <table class="form-table" role="presentation">
                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Enable protection', 'dcw-country-blocker-lite'); ?></th>
                        <td><label><input type="checkbox" name="dcw_cbl_settings[enabled]" value="1" <?php checked(1, (int) $o['enabled']); ?>> <?php echo esc_html__('Enabled', 'dcw-country-blocker-lite'); ?></label></td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Scope', 'dcw-country-blocker-lite'); ?></th>
                        <td>
                            <label><input type="radio" name="dcw_cbl_settings[scope]" value="login" <?php checked('login', $o['scope']); ?>> <?php echo esc_html__('Login only', 'dcw-country-blocker-lite'); ?></label><br>
                            <label><input type="radio" name="dcw_cbl_settings[scope]" value="admin_login" <?php checked('admin_login', $o['scope']); ?>> <?php echo esc_html__('Admin + Login', 'dcw-country-blocker-lite'); ?></label><br>
                            <label><input type="radio" name="dcw_cbl_settings[scope]" value="site" <?php checked('site', $o['scope']); ?>> <?php echo esc_html__('Entire site (front-end)', 'dcw-country-blocker-lite'); ?></label>
                        </td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Mode', 'dcw-country-blocker-lite'); ?></th>
                        <td>
                            <label><input type="radio" name="dcw_cbl_settings[mode]" value="block" <?php checked('block', $o['mode']); ?>> <?php echo esc_html__('Block listed countries', 'dcw-country-blocker-lite'); ?></label><br>
                            <label><input type="radio" name="dcw_cbl_settings[mode]" value="allow" <?php checked('allow', $o['mode']); ?>> <?php echo esc_html__('Allow only listed countries', 'dcw-country-blocker-lite'); ?></label>
                        </td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Countries (CSV, e.g. US, CA, DE)', 'dcw-country-blocker-lite'); ?></th>
                        <td><input type="text" class="regular-text" name="dcw_cbl_settings[countries]" value="<?php echo esc_attr($o['countries']); ?>" placeholder="US, CA, DE"></td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Admin bypass', 'dcw-country-blocker-lite'); ?></th>
                        <td><label><input type="checkbox" name="dcw_cbl_settings[admin_bypass]" value="1" <?php checked(1, (int) $o['admin_bypass']); ?>> <?php echo esc_html__('Admins always allowed (except preview)', 'dcw-country-blocker-lite'); ?></label></td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Allow IPs (CSV)', 'dcw-country-blocker-lite'); ?></th>
                        <td><input type="text" class="regular-text" name="dcw_cbl_settings[ip_allow]" value="<?php echo esc_attr($o['ip_allow']); ?>" placeholder="1.2.3.4, 5.6.7.8"></td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Action on deny', 'dcw-country-blocker-lite'); ?></th>
                        <td>
                            <label><input type="radio" name="dcw_cbl_settings[action]" value="blockpage" <?php checked('blockpage', $o['action']); ?>> <?php echo esc_html__('Show block page', 'dcw-country-blocker-lite'); ?></label><br>
                            <label><input type="radio" name="dcw_cbl_settings[action]" value="redirect" <?php checked('redirect', $o['action']); ?>> <?php echo esc_html__('Redirect to URL', 'dcw-country-blocker-lite'); ?></label>
                            <p>
                                <input type="url" class="regular-text" name="dcw_cbl_settings[redirect_url]" value="<?php echo esc_attr($o['redirect_url']); ?>" placeholder="https://example.com/denied">
                            </p>
                        </td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Block page — Title', 'dcw-country-blocker-lite'); ?></th>
                        <td><input type="text" class="regular-text" name="dcw_cbl_settings[title]" value="<?php echo esc_attr($o['title']); ?>"></td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Block page — Message (HTML allowed)', 'dcw-country-blocker-lite'); ?></th>
                        <td>
                            <?php
                            wp_editor(
                                wp_kses_post( $o['message'] ),
                                'dcw_cbl_message',
                                [
                                    'textarea_name' => 'dcw_cbl_settings[message]',
                                    'textarea_rows' => 5,
                                ]
                            );
                            ?>
                        </td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('Block page — Styling', 'dcw-country-blocker-lite'); ?></th>
                        <td>
                            <p>
                                <label><?php echo esc_html__('Background color', 'dcw-country-blocker-lite'); ?>
                                    <input type="text" name="dcw_cbl_settings[bg_color]" value="<?php echo esc_attr($o['bg_color']); ?>" class="dcw-cbl-color" data-default-color="#0f172a">
                                </label>
                            </p>
                            <p>
                                <label><?php echo esc_html__('Background image URL', 'dcw-country-blocker-lite'); ?>
                                    <input type="url" class="regular-text" name="dcw_cbl_settings[bg_image]" value="<?php echo esc_attr($o['bg_image']); ?>">
                                </label>
                            </p>
                            <p>
                                <label><?php echo esc_html__('Background size', 'dcw-country-blocker-lite'); ?>
                                    <select name="dcw_cbl_settings[bg_size]">
                                        <option value="cover" <?php selected('cover', $o['bg_size']); ?>><?php echo esc_html__('Cover (fill)', 'dcw-country-blocker-lite'); ?></option>
                                        <option value="contain" <?php selected('contain', $o['bg_size']); ?>><?php echo esc_html__('Contain (fit)', 'dcw-country-blocker-lite'); ?></option>
                                    </select>
                                </label>
                                &nbsp;&nbsp;
                                <label><?php echo esc_html__('Background position', 'dcw-country-blocker-lite'); ?>
                                    <select name="dcw_cbl_settings[bg_position]">
                                        <?php foreach (['top left','top center','top right','center left','center center','center right','bottom left','bottom center','bottom right'] as $pos): ?>
                                            <option value="<?php echo esc_attr($pos); ?>" <?php selected($pos, $o['bg_position']); ?>><?php echo esc_html(ucwords($pos)); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </label>
                            </p>
                            <p>
                                <label><?php echo esc_html__('Title color', 'dcw-country-blocker-lite'); ?>
                                    <input type="text" name="dcw_cbl_settings[title_color]" value="<?php echo esc_attr($o['title_color']); ?>" class="dcw-cbl-color" data-default-color="#ffffff">
                                </label>
                                &nbsp;&nbsp;
                                <label><?php echo esc_html__('Text color', 'dcw-country-blocker-lite'); ?>
                                    <input type="text" name="dcw_cbl_settings[text_color]" value="<?php echo esc_attr($o['text_color']); ?>" class="dcw-cbl-color" data-default-color="#e5e7eb">
                                </label>
                            </p>
                        </td>
                    </tr>

                    <tr class="dcw-cbl-field">
                        <th scope="row"><?php echo esc_html__('External lookup (optional)', 'dcw-country-blocker-lite'); ?></th>
                        <td>
                            <p class="description"><?php echo esc_html__('If your host/CDN does not set a country header, an external geolocation service can be used as a fallback.', 'dcw-country-blocker-lite'); ?></p>
                            <p>
                                <label><?php echo esc_html__('ipinfo.io Token', 'dcw-country-blocker-lite'); ?>
                                    <input type="text" class="regular-text" name="dcw_cbl_settings[ipinfo_token]" value="<?php echo esc_attr($o['ipinfo_token']); ?>" placeholder="">
                                </label>
                            </p>
                            <p>
                                <label><?php echo esc_html__('HTTP Timeout (seconds)', 'dcw-country-blocker-lite'); ?>
                                    <input type="number" min="1" step="1" name="dcw_cbl_settings[external_timeout]" value="<?php echo esc_attr( (string) (int) $o['external_timeout'] ); ?>">
                                </label>
                                &nbsp;&nbsp;
                                <label><?php echo esc_html__('If country unknown', 'dcw-country-blocker-lite'); ?>
                                    <select name="dcw_cbl_settings[api_failure_action]">
                                        <option value="allow" <?php selected('allow', $o['api_failure_action']); ?>><?php echo esc_html__('Allow', 'dcw-country-blocker-lite'); ?></option>
                                        <option value="block" <?php selected('block', $o['api_failure_action']); ?>><?php echo esc_html__('Block', 'dcw-country-blocker-lite'); ?></option>
                                    </select>
                                </label>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>

            <h2><?php echo esc_html__('Tools', 'dcw-country-blocker-lite'); ?></h2>
            <p class="dcw-cbl-actions">
                <a class="button button-secondary" target="_blank" href="<?php echo esc_url($preview_url); ?>">
                    <?php echo esc_html__('Preview block page', 'dcw-country-blocker-lite'); ?>
                </a>
                <a class="button button-secondary" target="_blank" href="<?php echo esc_url($diag_url); ?>">
                    <?php echo esc_html__('Open diagnostics', 'dcw-country-blocker-lite'); ?>
                </a>
                <span class="description">
                    <?php echo esc_html__('Diagnostics show your detected IP/CC and the decision path. Preview shows the block page without changing settings.', 'dcw-country-blocker-lite'); ?>
                </span>
            </p>
        </div>
        <?php
    }
}
