<?php
if ( ! defined('ABSPATH') ) exit;

if ( defined('WP_CLI') && WP_CLI ) {
    /**
     * Manage DCW Country Blocker Lite settings via WP-CLI.
     */
    class DCW_CBL_CLI {

        /** Get settings with defaults. */
        private static function get() {
            $o = get_option('dcw_cbl_settings', []);
            return wp_parse_args($o, [
                'enabled'      => 0,
                'mode'         => 'block',
                'scope'        => 'admin_login',
                'countries'    => '',
                'ip_allow'     => '',
                'admin_bypass' => 1,
                'action'       => 'blockpage',
                'redirect_url' => '',
            ]);
        }

        /** Save settings. */
        private static function save($o) {
            update_option('dcw_cbl_settings', $o);
        }

        /**
         * Enable protection.
         *
         * ## EXAMPLES
         *   wp dcw:cbl on
         */
        public function on( $args, $assoc_args ) {
            $o = self::get();
            $o['enabled'] = 1;
            self::save($o);
            \WP_CLI::success('DCW Country Blocker: enabled.');
        }

        /**
         * Disable protection.
         *
         * ## EXAMPLES
         *   wp dcw:cbl off
         */
        public function off( $args, $assoc_args ) {
            $o = self::get();
            $o['enabled'] = 0;
            self::save($o);
            \WP_CLI::success('DCW Country Blocker: disabled.');
        }

        /**
         * Set scope: login | admin_login | site
         *
         * ## OPTIONS
         * <scope>
         * : login, admin_login, or site
         *
         * ## EXAMPLES
         *   wp dcw:cbl scope admin_login
         */
        public function scope( $args, $assoc_args ) {
            list($scope) = $args;
            $scope = sanitize_key( $scope );

            $valid = ['login','admin_login','site'];
            if ( ! in_array($scope, $valid, true) ) {
                \WP_CLI::error('Invalid scope. Use: login, admin_login, or site.');
            }

            $o = self::get();
            $o['scope'] = $scope;
            self::save($o);

            \WP_CLI::success('Scope set to ' . $scope);
        }

        /**
         * Add an IP to the allow list.
         *
         * ## OPTIONS
         * <ip>
         *
         * ## EXAMPLES
         *   wp dcw:cbl add-ip 1.2.3.4
         */
        public function add_ip( $args, $assoc_args ) {
            list($ip) = $args;
            $ip = sanitize_text_field( $ip );

            $o   = self::get();
            $ips = array_filter( array_map('trim', explode(',', $o['ip_allow'])) );

            if ( ! in_array($ip, $ips, true) ) {
                $ips[] = $ip;
            }

            $o['ip_allow'] = implode(', ', $ips);
            self::save($o);

            \WP_CLI::success('Added ' . $ip);
        }

        /**
         * Remove an IP from the allow list.
         *
         * ## OPTIONS
         * <ip>
         *
         * ## EXAMPLES
         *   wp dcw:cbl rm-ip 1.2.3.4
         */
        public function rm_ip( $args, $assoc_args ) {
            list($ip) = $args;
            $ip = sanitize_text_field( $ip );

            $o   = self::get();
            $ips = array_filter( array_map('trim', explode(',', $o['ip_allow'])) );

            $ips = array_values(
                array_filter( $ips, function( $x ) use ( $ip ) {
                    return $x !== $ip;
                })
            );

            $o['ip_allow'] = implode(', ', $ips);
            self::save($o);

            \WP_CLI::success('Removed ' . $ip);
        }

        /**
         * Show current settings (brief).
         *
         * ## EXAMPLES
         *   wp dcw:cbl show
         */
        public function show( $args, $assoc_args ) {
            $o = self::get();
            \WP_CLI::line( wp_json_encode( $o, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES ) );
        }
    }

    \WP_CLI::add_command('dcw:cbl', 'DCW_CBL_CLI');
}
