<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_CBL_Frontend {

    public static function init(){
        add_action('init',              [__CLASS__, 'maybe_block_admin_area'], 0);
        add_action('template_redirect', [__CLASS__, 'maybe_block_site'],       0);
        add_action('template_redirect', [__CLASS__, 'maybe_diagnostics'],      0);
        add_action('template_redirect', [__CLASS__, 'maybe_admin_preview'],    0);
    }

    /* ===== Helpers ===== */

    private static function get_opts(){
        $o = get_option('dcw_cbl_settings', []);
        return wp_parse_args($o, [
            'enabled'            => 0,
            'scope'              => 'admin_login',
            'mode'               => 'block',
            'countries'          => '',
            'admin_bypass'       => 1,
            'ip_allow'           => '',
            'action'             => 'blockpage',
            'redirect_url'       => '',
            'title'              => 'Access Restricted',
            'message'            => '<p>Sorry, our site is not available in your region.</p>',
            'bg_color'           => '#0f172a',
            'bg_image'           => '',
            'bg_size'            => 'cover',
            'bg_position'        => 'center center',
            'title_color'        => '#ffffff',
            'text_color'         => '#e5e7eb',
            'preview_key'        => '',
            'preview_allow_one'  => 0,
            'ipinfo_token'       => '',
            'external_timeout'   => 3,
            'api_failure_action' => 'allow',
        ]);
    }

    private static function get_request_uri(){
        $uri = isset($_SERVER['REQUEST_URI']) ? wp_unslash( (string) $_SERVER['REQUEST_URI'] ) : '';
        return $uri;
    }

    private static function get_server_header($key){
        if ( empty($_SERVER[$key]) ) return '';
        $val = wp_unslash( (string) $_SERVER[$key] );
        return sanitize_text_field( $val );
    }

    private static function get_client_ip(){
        $keys = ['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP','REMOTE_ADDR'];
        foreach ($keys as $k){
            $raw = self::get_server_header($k);
            if ($raw === '') continue;
            $val = $raw;
            if ($k === 'HTTP_X_FORWARDED_FOR'){
                $parts = array_map('trim', explode(',', $val));
                $val = $parts[0] ?? '';
            }
            if ( filter_var($val, FILTER_VALIDATE_IP) ) {
                return $val;
            }
        }
        return self::get_server_header('REMOTE_ADDR');
    }

    private static function current_country_from_headers(){
        foreach (['HTTP_CF_IPCOUNTRY','HTTP_GEOIP_COUNTRY_CODE','GEOIP_COUNTRY_CODE'] as $h){
            $val = self::get_server_header($h);
            if ( $val !== '' ) {
                return strtoupper( substr($val, 0, 2) );
            }
        }
        return '';
    }

    private static function detect_country_external($opts){
        $ip = self::get_client_ip();
        if ( ! $ip ) return '';

        $token   = isset($opts['ipinfo_token']) ? trim( (string) $opts['ipinfo_token'] ) : '';
        $timeout = max(1, (int)($opts['external_timeout'] ?? 3));
        $tkey    = 'dcw_cbl_cc_' . md5($ip);

        $cached = get_transient($tkey);
        if ( $cached !== false ) {
            return strtoupper( substr( (string) $cached, 0, 2 ) );
        }

        if ( $token !== '' ){
            $url = 'https://ipinfo.io/' . rawurlencode($ip) . '/country?token=' . rawurlencode($token);
            $res = wp_remote_get($url, ['timeout'=>$timeout, 'sslverify'=>true, 'headers'=>['Accept'=>'text/plain']]);
            if ( ! is_wp_error($res) && wp_remote_retrieve_response_code($res) === 200 ){
                $body = trim( (string) wp_remote_retrieve_body($res) );
                if ( $body !== '' ){
                    set_transient($tkey, $body, DAY_IN_SECONDS);
                    return strtoupper( substr($body, 0, 2) );
                }
            }
        }

        $url = 'https://ipapi.co/' . rawurlencode($ip) . '/json/';
        $res = wp_remote_get($url, ['timeout'=>$timeout, 'sslverify'=>true, 'headers'=>['Accept'=>'application/json']]);
        if ( ! is_wp_error($res) && wp_remote_retrieve_response_code($res) === 200 ){
            $data = json_decode( (string) wp_remote_retrieve_body($res), true );
            if ( is_array($data) && ! empty($data['country']) ){
                set_transient($tkey, $data['country'], DAY_IN_SECONDS);
                return strtoupper( substr( (string) $data['country'], 0, 2 ) );
            }
        }

        set_transient($tkey, '', 3 * MINUTE_IN_SECONDS);
        return '';
    }

    private static function resolved_country($opts){
        if ( isset($_GET['dcw_cbl_force_cc']) && isset($_GET['_dcw_cbl_nonce']) ) {
            $nonce = sanitize_text_field( wp_unslash( (string) $_GET['_dcw_cbl_nonce'] ) );
            if ( current_user_can('manage_options') && wp_verify_nonce($nonce, 'dcw_cbl_diag') ) {
                $f = strtoupper( substr( sanitize_text_field( wp_unslash( (string) $_GET['dcw_cbl_force_cc'] ) ), 0, 2 ) );
                if ( preg_match('/^[A-Z]{2}$/', $f) ) {
                    return $f;
                }
            }
        }

        $cc = self::current_country_from_headers();
        if ( $cc !== '' ) return $cc;

        return self::detect_country_external($opts);
    }

    private static function has_preview_cookie(){
        return ! empty($_COOKIE['dcw_cbl_preview']) && is_string($_COOKIE['dcw_cbl_preview']);
    }

    /**
     * Recognize the actual login endpoint even if a plugin changes it.
     */
    private static function is_login_request(){
        $uri = isset($_SERVER['REQUEST_URI']) ? wp_unslash((string) $_SERVER['REQUEST_URI']) : '';
        $current_path = parse_url( home_url( $uri ), PHP_URL_PATH );

        $login_url  = wp_login_url();
        $login_path = parse_url( $login_url, PHP_URL_PATH );

        if ( $login_path && $current_path && rtrim($current_path,'/') === rtrim($login_path,'/') ) {
            return true;
        }

        // Fallback for classic
        return (strpos($uri, 'wp-login.php') !== false);
    }

    /* ===== Core blocking switches ===== */

    public static function maybe_block_admin_area(){
        $opts = self::get_opts();
        if ( empty($opts['enabled']) ) return;

        $scope = $opts['scope'];
        if ( $scope === 'site' ) return;

        if ( wp_doing_ajax() || wp_doing_cron() ) return;

        $uri      = self::get_request_uri();
        $is_login = self::is_login_request();
        $is_admin = ( strpos($uri, '/wp-admin') === 0 );

        $target = false;
        if ( $scope === 'login' ) {
            $target = $is_login;
        } elseif ( $scope === 'admin_login' ) {
            $target = ( $is_login || $is_admin );
        }

        if ( ! $target ) return;
        self::maybe_block_now($opts);
    }

    public static function maybe_block_site(){
        $opts = self::get_opts();
        if ( empty($opts['enabled']) ) return;
        if ( ($opts['scope'] ?? 'admin_login') !== 'site' ) return;

        if ( wp_doing_ajax() || wp_doing_cron() ) return;

        $uri = self::get_request_uri();
        if ( strpos($uri, '/wp-json') === 0 ) return;
        if ( self::is_login_request() ) return;
        if ( strpos($uri, '/wp-admin') === 0 ) return;

        if ( ! empty($opts['admin_bypass']) && current_user_can('manage_options') ) return;

        self::maybe_block_now($opts);
    }

    private static function maybe_block_now($opts){
        if ( ! empty($opts['admin_bypass']) && current_user_can('manage_options') ) return;

        $ip = self::get_client_ip();
        if ( $ip && ! empty($opts['ip_allow']) ) {
            $ips = array_filter( array_map('trim', explode(',', (string) $opts['ip_allow'])) );
            if ( in_array($ip, $ips, true) ) return;
        }

        if ( self::has_preview_cookie() ) return;

        $country = strtoupper( substr( (string) self::resolved_country($opts), 0, 2 ) );

        $mode = $opts['mode'] ?? 'block';
        $csv  = $opts['countries'] ?? '';
        $list = array_filter( array_map('trim', explode(',', strtoupper( (string) $csv ))) );
        $in_list = in_array($country, $list, true);

        $deny = false;

        if ( $country === '' ) {
            $policy = $opts['api_failure_action'] ?? 'allow';
            $deny = ( $policy === 'block' );
        } else {
            if ( $mode === 'block' ) {
                $deny = $in_list;          // block listed
            } else {
                $deny = ! $in_list;        // allow only listed
            }
        }

        if ( ! $deny ) return;

        $action = $opts['action'] ?? 'blockpage';
        if ( $action === 'redirect' && ! empty($opts['redirect_url']) ) {
            wp_safe_redirect( esc_url_raw( $opts['redirect_url'] ) );
            exit;
        }

        status_header(403);
        nocache_headers();
        header('X-Robots-Tag: noindex, nofollow', true);
        self::render_block_page($opts);
        exit;
    }

    public static function maybe_admin_preview(){
        if ( empty($_GET['dcw_cbl_preview']) || empty($_GET['_dcw_cbl_nonce']) ) return;

        $flag  = sanitize_text_field( wp_unslash( (string) $_GET['dcw_cbl_preview'] ) );
        $nonce = sanitize_text_field( wp_unslash( (string) $_GET['_dcw_cbl_nonce'] ) );

        if ( $flag !== '1' ) return;
        if ( ! current_user_can('manage_options') ) return;
        if ( ! wp_verify_nonce($nonce, 'dcw_cbl_preview') ) return;

        $opts = self::get_opts();
        status_header(403);
        nocache_headers();
        header('X-Robots-Tag: noindex, nofollow', true);
        self::render_block_page($opts);
        exit;
    }

    public static function maybe_diagnostics(){
        if ( empty($_GET['dcw_cbl_diag']) ) return;

        if ( empty($_GET['_dcw_cbl_nonce']) ) return;
        $nonce = sanitize_text_field( wp_unslash( (string) $_GET['_dcw_cbl_nonce'] ) );
        if ( ! current_user_can('manage_options') || ! wp_verify_nonce($nonce, 'dcw_cbl_diag') ) {
            return;
        }

        $opts = self::get_opts();

        $as_guest = ( ! empty($_GET['dcw_cbl_as_guest']) && sanitize_text_field( wp_unslash( (string) $_GET['dcw_cbl_as_guest'] ) ) === '1' );

        $uri   = self::get_request_uri();
        $ip    = self::get_client_ip();
        $hcc   = self::current_country_from_headers();
        $acc   = self::detect_country_external($opts);
        $cc    = $acc ?: $hcc;

        $enabled   = ! empty($opts['enabled']) ? 'yes' : 'no';
        $scope     = $opts['scope'] ?? 'admin_login';
        $mode      = $opts['mode']  ?? 'block';
        $countries = strtoupper( (string) ($opts['countries'] ?? '') );

        $preview_cookie = self::has_preview_cookie() ? 'yes' : 'no';
        $admin_bypass   = ! empty($opts['admin_bypass']);
        $user_is_admin  = current_user_can('manage_options');

        $list     = array_filter( array_map('trim', explode(',', $countries)) );
        $in_list  = in_array( strtoupper( substr( (string) $cc, 0, 2 ) ), $list, true );
        $decision = 'ALLOW';
        $reason   = 'mode_allow_listed';

        if ( $cc === '' ) {
            $policy  = $opts['api_failure_action'] ?? 'allow';
            $deny    = ( $policy === 'block' );
            $decision= $deny ? 'BLOCK' : 'ALLOW';
            $reason  = $deny ? 'unknown_block_policy' : 'unknown_allow_policy';
        } else {
            if ( $mode === 'block' ) {
                $deny    = $in_list;
                $decision= $deny ? 'BLOCK' : 'ALLOW';
                $reason  = $deny ? 'mode_block_listed' : 'mode_block_not_listed';
            } else {
                $deny    = ! $in_list;
                $decision= $deny ? 'BLOCK' : 'ALLOW';
                $reason  = $deny ? 'mode_allow_not_listed' : 'mode_allow_listed';
            }
        }

        if ( ! $as_guest ) {
            if ( $admin_bypass && $user_is_admin ) {
                $decision = 'ALLOW';
                $reason   = 'admin_bypass';
            }
            if ( self::has_preview_cookie() ) {
                $decision = 'ALLOW';
                $reason   = 'preview_cookie';
            }
        }

        header('Content-Type: text/plain; charset=utf-8');
        echo esc_html( "DCW Country Blocker Lite — Diagnostics\n" );
        echo esc_html( str_repeat('-', 39) . "\n" );
        echo esc_html( 'Request URI: ' . $uri . "\n" );
        echo esc_html( 'Client IP:   ' . $ip  . "\n" );
        echo esc_html( 'Header CC:   ' . $hcc . "\n" );
        echo esc_html( 'API CC:      ' . $acc . "\n\n" );

        echo esc_html( 'Enabled:         ' . $enabled . "\n" );
        echo esc_html( 'Scope:           ' . $scope . "\n" );
        echo esc_html( 'Mode:            ' . $mode  . "\n" );
        echo esc_html( 'Countries:       ' . $countries . "\n" );
        echo esc_html( 'Preview cookie:  ' . $preview_cookie . "  (add ?dcw_cbl_clear_preview=1&_dcw_cbl_nonce=... to clear)\n" );
        echo esc_html( 'admin_bypass opt:' . ( $admin_bypass ? 'on' : 'off') . "\n" );
        echo esc_html( 'user_is_admin:   ' . ( $user_is_admin ? 'yes' : 'no' ) . "\n" );
        echo esc_html( 'as_guest:        ' . ( $as_guest ? 'yes' : 'no' ) . "\n" );
        echo esc_html( 'Resolved CC:     ' . ( $cc ?: '(unknown)' ) . "\n" );
        echo esc_html( 'Decision:        ' . $decision . '  via ' . $reason . "\n" );
        exit;
    }

    private static function maybe_clear_preview_cookie(){
        if ( empty($_GET['dcw_cbl_clear_preview']) || empty($_GET['_dcw_cbl_nonce']) ) return false;
        $nonce = sanitize_text_field( wp_unslash( (string) $_GET['_dcw_cbl_nonce'] ) );
        if ( ! current_user_can('manage_options') || ! wp_verify_nonce($nonce, 'dcw_cbl_diag') ) {
            return false;
        }
        @setcookie('dcw_cbl_preview', '', time() - 3600, COOKIEPATH ? COOKIEPATH : '/', COOKIE_DOMAIN, is_ssl(), true);
        unset($_COOKIE['dcw_cbl_preview']);
        return true;
    }

    private static function render_block_page($opts){
        self::maybe_clear_preview_cookie();

        $title   = $opts['title']   ?? 'Access Restricted';
        $message = $opts['message'] ?? '<p>Sorry, our site is not available in your region.</p>';
        $bg      = $opts['bg_color']    ?? '#0f172a';
        $bg_img  = $opts['bg_image']    ?? '';
        $bg_size = $opts['bg_size']     ?? 'cover';
        $bg_pos  = $opts['bg_position'] ?? 'center center';
        $title_c = $opts['title_color'] ?? '#ffffff';
        $text_c  = $opts['text_color']  ?? '#e5e7eb';

        $inline  = '--dcw-bg:' . esc_attr($bg) . '; --dcw-title:' . esc_attr($title_c) . '; --dcw-text:' . esc_attr($text_c) . ';';
        if ( ! empty($bg_img) ){
            $inline .= ' background-image:url(' . esc_url($bg_img) . '); background-size:' . esc_attr($bg_size) . '; background-position:' . esc_attr($bg_pos) . '; background-repeat:no-repeat;';
        }
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <meta name="robots" content="noindex,nofollow">
            <title><?php echo esc_html($title); ?></title>
            <?php wp_head(); ?>
        </head>
        <body class="dcw-cbl-block" style="<?php echo esc_attr($inline); ?>">
            <main class="dcw-wrap">
                <h1 class="dcw-title" style="color:var(--dcw-title);"><?php echo esc_html($title); ?></h1>
                <div class="dcw-msg"  style="color:var(--dcw-text);"><?php echo wp_kses_post($message); ?></div>
            </main>
            <?php wp_footer(); ?>
        </body>
        </html>
        <?php
    }
}
