<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_FHL_Admin {

    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'menu' ] );
        add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );
    }

    public static function menu() {
        add_options_page(
            esc_html__( 'DCW Force HTTPS Lite', 'dcw-force-https-lite' ),
            esc_html__( 'DCW Force HTTPS', 'dcw-force-https-lite' ),
            'manage_options',
            'dcw-fhl',
            [ __CLASS__, 'settings_page' ]
        );
    }

    public static function register_settings() {
        register_setting(
            'dcw_fhl_group',
            'dcw_fhl_options',
            [
                'type'              => 'array',
                'sanitize_callback' => [ __CLASS__, 'sanitize' ],
                'default'           => [],
            ]
        );
    }

    public static function sanitize( $in ) {
        if ( ! is_array( $in ) ) {
            $in = [];
        }
        $out = [];
        $out['enabled'] = ! empty( $in['enabled'] ) ? 1 : 0;

        // List of page IDs to EXCLUDE from forcing HTTPS (optional feature)
        $out['exclude_pages'] = [];
        if ( ! empty( $in['exclude_pages'] ) && is_array( $in['exclude_pages'] ) ) {
            $out['exclude_pages'] = array_values( array_unique( array_map( 'intval', $in['exclude_pages'] ) ) );
        }

        return $out;
    }

    /**
     * VIP-safe page fetcher:
     * Build an allowed/whitelist set FIRST, then query with post__in (no exclude / post__not_in).
     *
     * @param int[] $exclude_ids
     * @return WP_Post[]
     */
    private static function get_pages_whitelist( $exclude_ids = [] ) {
        $exclude_ids = array_map( 'intval', (array) $exclude_ids );

        // 1) Get all published page IDs (IDs only; fast)
        $all_ids = get_posts( [
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'no_found_rows'  => true,
            'orderby'        => 'none',
            'posts_per_page' => -1,
        ] );

        if ( empty( $all_ids ) ) {
            return [];
        }

        // 2) Compute whitelist (allowed set = all minus excluded)
        $allowed_ids = array_values( array_diff( array_map( 'intval', $all_ids ), $exclude_ids ) );

        if ( empty( $allowed_ids ) ) {
            return [];
        }

        // 3) Fetch the actual posts using post__in (VIP-safe) and sort by title
        $pages = get_posts( [
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'post__in'       => $allowed_ids,
            'no_found_rows'  => true,
            'orderby'        => 'title',
            'order'          => 'ASC',
            'posts_per_page' => -1,
        ] );

        return $pages ?: [];
    }

    /**
     * Helper: fetch EXACT list of pages by IDs (used to display current excludes).
     *
     * @param int[] $ids
     * @return WP_Post[]
     */
    private static function get_pages_by_ids( $ids ) {
        $ids = array_values( array_unique( array_map( 'intval', (array) $ids ) ) );
        if ( empty( $ids ) ) return [];

        return get_posts( [
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'post__in'       => $ids,
            'no_found_rows'  => true,
            'orderby'        => 'title',
            'order'          => 'ASC',
            'posts_per_page' => -1,
        ] ) ?: [];
    }

    public static function settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $opts = get_option( 'dcw_fhl_options', [] );
        $opts = wp_parse_args( $opts, [
            'enabled'       => 1,
            'exclude_pages' => [], // optional: pages where HTTPS should NOT be forced
        ] );

        $exclude_ids = array_map( 'intval', (array) $opts['exclude_pages'] );

        // VIP-safe: get all *allowed* pages WITHOUT using exclude/post__not_in.
        $allowed_pages  = self::get_pages_whitelist( $exclude_ids );
        // Pages currently excluded (shown under the selector)
        $excluded_pages = self::get_pages_by_ids( $exclude_ids );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'DCW Force HTTPS Lite', 'dcw-force-https-lite' ); ?></h1>

            <form method="post" action="options.php">
                <?php settings_fields( 'dcw_fhl_group' ); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="dcw_fhl_enabled"><?php echo esc_html__( 'Enable force HTTPS', 'dcw-force-https-lite' ); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" id="dcw_fhl_enabled" name="dcw_fhl_options[enabled]" value="1" <?php checked( 1, (int) $opts['enabled'] ); ?> />
                            <p class="description"><?php echo esc_html__( 'Redirect all front-end HTTP requests to HTTPS.', 'dcw-force-https-lite' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="dcw_fhl_exclude_pages"><?php echo esc_html__( 'Exclude pages (optional)', 'dcw-force-https-lite' ); ?></label>
                        </th>
                        <td>
                            <select id="dcw_fhl_exclude_pages" name="dcw_fhl_options[exclude_pages][]" multiple="multiple" style="min-width:320px; height: 12em;">
                                <?php
                                // Show allowed pages as options; selected ones are those already excluded.
                                foreach ( $allowed_pages as $p ) {
                                    $id    = (int) $p->ID;
                                    $title = get_the_title( $p );

                                    // Build the display label, with translators comment for the placeholder.
                                    if ( $title === '' ) {
                                        /* translators: %d: page ID */
                                        $label = sprintf( __( 'Untitled (#%d)', 'dcw-force-https-lite' ), $id );
                                    } else {
                                        $label = $title;
                                    }

                                    printf(
                                        '<option value="%s"%s>%s</option>',
                                        esc_attr( (string) $id ),
                                        selected( in_array( $id, $exclude_ids, true ), true, false ),
                                        esc_html( $label )
                                    );
                                }
                                ?>
                            </select>
                            <p class="description"><?php echo esc_html__( 'Hold Ctrl/Cmd to select multiple. Selected pages will NOT be forced to HTTPS.', 'dcw-force-https-lite' ); ?></p>
                        </td>
                    </tr>

                    <?php if ( ! empty( $excluded_pages ) ) : ?>
                        <tr>
                            <th scope="row"><?php echo esc_html__( 'Currently excluded', 'dcw-force-https-lite' ); ?></th>
                            <td>
                                <ul style="margin:0;">
                                    <?php foreach ( $excluded_pages as $p ) : ?>
                                        <li>
                                            <?php
                                            $title = get_the_title( $p );
                                            if ( $title === '' ) {
                                                /* translators: %d: page ID */
                                                $label = sprintf( __( 'Untitled (#%d)', 'dcw-force-https-lite' ), (int) $p->ID );
                                            } else {
                                                $label = $title;
                                            }
                                            echo esc_html( $label );
                                            ?>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            </td>
                        </tr>
                    <?php endif; ?>
                </table>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}
