<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_FHL_Core {

    public static function init() {
        // Force HTTPS as early as possible on front-end requests.
        add_action('template_redirect', [__CLASS__, 'maybe_force_https'], 0);
    }

    /**
     * Detect scheme via is_ssl() and proxy/CDN headers, then redirect to HTTPS if needed.
     */
    public static function maybe_force_https() {
        // Don’t interfere with admin-ajax, cron, CLI, or already-secure requests.
        if ( is_admin() ) {
            return;
        }

        // 1) Quick pass: WordPress’ native SSL detection.
        $is_secure = is_ssl();

        // 2) Consider reverse-proxy headers (properly sanitized/unslashed).
        if ( ! $is_secure ) {
            $xfp = '';
            if ( isset($_SERVER['HTTP_X_FORWARDED_PROTO']) ) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $xfp = sanitize_text_field( wp_unslash( (string) $_SERVER['HTTP_X_FORWARDED_PROTO'] ) );
            }
            if ( $xfp ) {
                // Some proxies send "https,http" – check if it contains "https".
                if ( strpos( strtolower( $xfp ), 'https' ) !== false ) {
                    $is_secure = true;
                }
            }
        }

        if ( ! $is_secure ) {
            $cf_scheme = '';
            if ( isset($_SERVER['HTTP_CF_VISITOR']) ) {
                // Cloudflare sends JSON: {"scheme":"https"}
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $raw_cfv = wp_unslash( (string) $_SERVER['HTTP_CF_VISITOR'] );
                $cfv     = json_decode( $raw_cfv, true );
                if ( is_array($cfv) && ! empty($cfv['scheme']) ) {
                    $cf_scheme = sanitize_text_field( (string) $cfv['scheme'] );
                }
            }
            if ( $cf_scheme === 'https' ) {
                $is_secure = true;
            }
        }

        // Already secure? Nothing to do.
        if ( $is_secure ) {
            return;
        }

        // 3) Build HTTPS URL safely.
        $host = '';
        if ( isset($_SERVER['HTTP_HOST']) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $host = sanitize_text_field( wp_unslash( (string) $_SERVER['HTTP_HOST'] ) );
        } elseif ( isset($_SERVER['SERVER_NAME']) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $host = sanitize_text_field( wp_unslash( (string) $_SERVER['SERVER_NAME'] ) );
        }

        $uri = '/';
        if ( isset($_SERVER['REQUEST_URI']) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $uri = sanitize_text_field( wp_unslash( (string) $_SERVER['REQUEST_URI'] ) );
            if ( $uri === '' ) {
                $uri = '/';
            }
        }

        // If we still can't determine host, bail safely.
        if ( $host === '' ) {
            return;
        }

        // Force 301 to HTTPS.
        $target = 'https://' . $host . $uri;

        // Use wp_safe_redirect; allow only same-host (which we’re constructing).
        wp_safe_redirect( $target, 301 );
        exit;
    }
}
