<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class DCW_LLL_Admin {

    private static $hook = '';

    public static function init(){
        add_action( 'admin_menu', [__CLASS__, 'menu'] );
        add_action( 'admin_init', [__CLASS__, 'register_settings'] );
        add_action( 'admin_enqueue_scripts', [__CLASS__, 'assets'] );
        add_action( 'load-settings_page_dcw-lll', [__CLASS__, 'add_help_tabs'] );
    }

    public static function menu(){
        self::$hook = add_options_page(
            __('DCW Login Limit Lite','dcw-login-limit-lite'),
            __('DCW Login Limit Lite','dcw-login-limit-lite'),
            'manage_options',
            'dcw-lll',
            [__CLASS__, 'page']
        );
    }

    public static function assets( $hook ){
        if ( $hook !== self::$hook ) return;
        wp_enqueue_style( 'dcw-lll-admin', DCW_LLL_URL . 'assets/css/admin.css', [], DCW_LLL_VER );
        wp_enqueue_script( 'dcw-lll-admin', DCW_LLL_URL . 'assets/js/admin.js', ['jquery'], DCW_LLL_VER, true );
    }

    public static function register_settings(){
        register_setting( 'dcw_lll', 'dcw_lll_options', [__CLASS__, 'sanitize'] );
    }

    public static function defaults(){
        return [
            'enabled'             => 1,
            'max_attempts'        => 5,
            'lockout_time'        => 30,
            'whitelist'           => '',
            'message'             => __('Too many failed login attempts. Please try again later.','dcw-login-limit-lite'),
            'delete_on_uninstall' => 0,
            'enable_update_check' => 1,
        ];
    }

    public static function get(){
        $o = get_option('dcw_lll_options', []);
        return wp_parse_args( $o, self::defaults() );
    }

    public static function sanitize( $in ){
        $d = self::defaults();
        $out = [];
        $out['enabled']      = ! empty($in['enabled']) ? 1 : 0;
        $out['max_attempts'] = max(1, (int)($in['max_attempts'] ?? $d['max_attempts']));
        $out['lockout_time'] = max(1, (int)($in['lockout_time'] ?? $d['lockout_time']));

        $wl = isset($in['whitelist']) ? (string)$in['whitelist'] : '';
        $wl = preg_replace('/[^0-9a-fA-F\:\.\,\s]/', '', $wl);
        $out['whitelist']    = trim($wl);

        $out['message']      = isset($in['message']) ? wp_kses_post($in['message']) : $d['message'];
        $out['delete_on_uninstall'] = ! empty($in['delete_on_uninstall']) ? 1 : 0;
        $out['enable_update_check'] = ! empty($in['enable_update_check']) ? 1 : 0;
        return $out;
    }

    private static function tip( $text ){
        echo '<span class="dcw-help" tabindex="0" aria-label="' . esc_attr( wp_strip_all_tags($text) ) . '">?<span class="dcw-tip">'. wp_kses_post($text) .'</span></span>';
    }

    public static function add_help_tabs(){
        $screen = get_current_screen();
        if ( ! $screen || $screen->id !== 'settings_page_dcw-lll' ) return;

        $screen->add_help_tab([
            'id'    => 'dcw_lll_overview',
            'title' => __('Overview','dcw-login-limit-lite'),
            'content' => '<p>' . esc_html__('Limit failed login attempts and lock abusive IPs to protect wp-login.php.', 'dcw-login-limit-lite') . '</p>'
        ]);
        $screen->add_help_tab([
            'id'    => 'dcw_lll_detection',
            'title' => __('Detection','dcw-login-limit-lite'),
            'content' => '<p>' . esc_html__('The plugin reads CF-Connecting-IP / X-Forwarded-For when present, otherwise REMOTE_ADDR. Whitelisted IPs bypass limits.', 'dcw-login-limit-lite') . '</p>'
        ]);
        $screen->add_help_tab([
            'id'    => 'dcw_lll_troubleshooting',
            'title' => __('Troubleshooting','dcw-login-limit-lite'),
            'content' => '<ul><li>' . esc_html__('Locked out? Wait for lockout to expire or log in from a whitelisted IP.', 'dcw-login-limit-lite') . '</li><li>' . esc_html__('Behind a proxy/CDN? Ensure real client IP headers are forwarded.', 'dcw-login-limit-lite') . '</li></ul>'
        ]);
        $screen->add_help_tab([
            'id'    => 'dcw_lll_bypass',
            'title' => __('Bypass & Whitelist','dcw-login-limit-lite'),
            'content' => '<p>' . esc_html__('Add trusted addresses to Whitelist IPs to bypass limits during maintenance or admin access.', 'dcw-login-limit-lite') . '</p>'
        ]);
        $screen->set_help_sidebar('<p><strong>' . esc_html__('Need help?', 'dcw-login-limit-lite') . '</strong><br>' . esc_html__('Contact DCWireNet support.', 'dcw-login-limit-lite') . '</p>');
    }

    public static function page(){
        if ( ! current_user_can('manage_options') ) return;
        $o = self::get();
        ?>
        <div class="wrap dcw-lll-wrap">
            <h1><?php esc_html_e('DCW Login Limit Lite','dcw-login-limit-lite'); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields('dcw_lll'); ?>
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Enable protection','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('Turn the limiter on or off. When disabled, no rate limiting is applied.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <label><input type="checkbox" name="dcw_lll_options[enabled]" value="1" <?php checked(1, !empty($o['enabled'])); ?>> <?php esc_html_e('Active','dcw-login-limit-lite'); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Max failed attempts','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('Failed attempts allowed before locking an IP.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <input type="number" min="1" name="dcw_lll_options[max_attempts]" value="<?php echo esc_attr($o['max_attempts']); ?>" class="small-text">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Lockout (minutes)','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('How long the IP is blocked after reaching the limit.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <input type="number" min="1" name="dcw_lll_options[lockout_time]" value="<?php echo esc_attr($o['lockout_time']); ?>" class="small-text">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Whitelist IPs','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('Comma-separated list. These IPs are never blocked.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <input type="text" name="dcw_lll_options[whitelist]" value="<?php echo esc_attr($o['whitelist']); ?>" class="regular-text" placeholder="127.0.0.1, 10.0.0.1">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Lockout message','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('Shown on wp-login.php when an IP is temporarily blocked.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <textarea name="dcw_lll_options[message]" rows="3" class="large-text"><?php echo esc_textarea($o['message']); ?></textarea>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Delete data on uninstall','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('When enabled, plugin settings are erased when you uninstall.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <label><input type="checkbox" name="dcw_lll_options[delete_on_uninstall]" value="1" <?php checked(1, !empty($o['delete_on_uninstall'])); ?>> <?php esc_html_e('Delete all plugin data on uninstall','dcw-login-limit-lite'); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Enable update checks','dcw-login-limit-lite'); ?>
                                <?php self::tip(__('When enabled, the plugin checks dcwirenet.com for new versions using a JSON endpoint.','dcw-login-limit-lite')); ?>
                            </th>
                            <td>
                                <label><input type="checkbox" name="dcw_lll_options[enable_update_check]" value="1" <?php checked(1, !empty($o['enable_update_check'])); ?>> <?php esc_html_e('Check for updates automatically','dcw-login-limit-lite'); ?></label>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}
