<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_LLL_Frontend {

    public static function init(){
        add_action( 'wp_login_failed', [__CLASS__, 'on_login_failed'] );
        add_filter( 'authenticate', [__CLASS__, 'auth_block_if_locked'], 30, 3 );
        add_action( 'wp_login', [__CLASS__, 'on_login_success'], 10, 2 );
    }

    private static function opts(){
        $d = [
            'enabled'      => 1,
            'max_attempts' => 5,
            'lockout_time' => 30,
            'whitelist'    => '',
            'message'      => __('Too many failed login attempts. Please try again later.','dcw-login-limit-lite'),
        ];
        $o = get_option('dcw_lll_options', []);
        return wp_parse_args($o, $d);
    }

    private static function client_ip(){
        $keys = ['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP','REMOTE_ADDR'];
        foreach ( $keys as $k ){
            if ( ! empty($_SERVER[$k]) ){
                $v = $_SERVER[$k];
                if ( $k === 'HTTP_X_FORWARDED_FOR' ){
                    $parts = array_map('trim', explode(',', $v));
                    $v = $parts[0];
                }
                if ( filter_var($v, FILTER_VALIDATE_IP) ) return $v;
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }

    private static function is_whitelisted($ip, $o){
        $ips = array_filter(array_map('trim', explode(',', (string)$o['whitelist'])));
        return in_array($ip, $ips, true);
    }

    private static function k_attempts($ip){ return 'dcw_lll_' . md5($ip); }
    private static function k_blocked($ip){ return 'dcw_lll_blocked_' . md5($ip); }

    public static function on_login_failed($username){
        $o = self::opts();
        if ( empty($o['enabled']) ) return;
        $ip = self::client_ip();
        if ( self::is_whitelisted($ip, $o) ) return;

        $kA = self::k_attempts($ip);
        $kB = self::k_blocked($ip);

        $attempts = (int) get_transient($kA);
        $attempts++;
        set_transient($kA, $attempts, max(60, (int)$o['lockout_time']*MINUTE_IN_SECONDS));

        if ( $attempts >= (int)$o['max_attempts'] ){
            set_transient($kB, 1, max(60, (int)$o['lockout_time']*MINUTE_IN_SECONDS));
            delete_transient($kA);
        }
    }

    public static function auth_block_if_locked($user, $username, $password){
        $o = self::opts();
        if ( empty($o['enabled']) ) return $user;

        $ip = self::client_ip();
        if ( self::is_whitelisted($ip, $o) ) return $user;

        if ( get_transient(self::k_blocked($ip)) ){
            return new WP_Error('dcw_lll_locked', (string)$o['message']);
        }
        return $user;
    }

    public static function on_login_success($user_login, $user){
        $o = self::opts();
        if ( empty($o['enabled']) ) return;
        $ip = self::client_ip();
        delete_transient(self::k_attempts($ip));
        // keep lock until it naturally expires for safety
    }
}
