<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_MML_Admin {

    /** @var string */
    private static $hook = '';

    public static function init(){
        add_action( 'admin_menu', [__CLASS__, 'menu'] );
        add_action( 'admin_init', [__CLASS__, 'register_settings'] );
        add_action( 'admin_enqueue_scripts', [__CLASS__, 'assets'] );

        // Purge caches when our option updates and the toggle flips.
        add_action( 'updated_option', [__CLASS__, 'maybe_purge_on_toggle'], 10, 3 );
    }

    public static function menu() {
        $hook = add_options_page(
            esc_html__( 'DCW Maintenance Mode', 'dcw-maintenance-mode-lite' ),
            esc_html__( 'DCW Maintenance Mode', 'dcw-maintenance-mode-lite' ),
            'manage_options',
            'dcw-mml',
            [__CLASS__, 'settings_page']
        );
        self::$hook = $hook;

        // Add help tabs when this screen is loaded.
        add_action( 'load-' . $hook, [__CLASS__, 'add_help_tabs'] );
    }

    public static function register_settings() {
        register_setting('dcw_mml_group', 'dcw_mml_settings', [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
            'default'           => []
        ]);
    }

    public static function sanitize($input) {
        if ( ! is_array($input) ) {
            $input = [];
        }
        $out = [];
        $out['enabled']      = !empty($input['enabled']) ? 1 : 0;
        $out['title']        = isset($input['title']) ? sanitize_text_field( wp_unslash( $input['title'] ) ) : '';
        $out['message']      = isset($input['message']) ? wp_kses_post( $input['message'] ) : '';
        $out['logo']         = isset($input['logo']) ? esc_url_raw( $input['logo'] ) : '';

        $out['bg_color']     = isset($input['bg_color']) ? ( sanitize_hex_color( $input['bg_color'] ) ?: '#0f172a' ) : '#0f172a';
        $out['bg_image']     = isset($input['bg_image']) ? esc_url_raw( $input['bg_image'] ) : '';
        $out['bg_size']      = ( isset($input['bg_size']) && in_array($input['bg_size'], ['cover','contain'], true) ) ? $input['bg_size'] : 'cover';
        $positions = ['top left','top center','top right','center left','center center','center right','bottom left','bottom center','bottom right'];
        $out['bg_position']  = ( isset($input['bg_position']) && in_array($input['bg_position'], $positions, true) ) ? $input['bg_position'] : 'center center';

        $out['title_color']  = isset($input['title_color']) ? ( sanitize_hex_color( $input['title_color'] ) ?: '#ffffff' ) : '#ffffff';
        $out['text_color']   = isset($input['text_color'])  ? ( sanitize_hex_color( $input['text_color'] )  ?: '#e5e7eb' ) : '#e5e7eb';

        $out['countdown_enable'] = !empty($input['countdown_enable']) ? 1 : 0;
        $out['end_datetime']     = isset($input['end_datetime']) ? sanitize_text_field( wp_unslash( $input['end_datetime'] ) ) : '';

        // Preview bypass settings.
        $out['preview_key']       = isset($input['preview_key']) ? sanitize_text_field( wp_unslash( $input['preview_key'] ) ) : '';
        $out['preview_allow_one'] = !empty($input['preview_allow_one']) ? 1 : 0;

        return $out;
    }

    public static function assets($hook) {
        if ( $hook !== 'settings_page_dcw-mml' ) return;
        wp_enqueue_media();
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');

        // Admin CSS (for tooltips, minor styles).
        wp_enqueue_style('dcw-mml-admin-css', DCW_MML_URL . 'assets/css/admin.css', [], DCW_MML_VERSION);
        // Admin JS (media button, color picker init).
        wp_enqueue_script('dcw-mml-admin', DCW_MML_URL . 'assets/js/admin.js', ['jquery','wp-color-picker'], DCW_MML_VERSION, true);
    }

    /** Small "?" tooltip helper. */
    private static function tip( $text ){
        return '<span class="dcw-help" tabindex="0" aria-label="' . esc_attr( $text ) . '" data-tip="' . esc_attr( $text ) . '">?</span>';
    }

    public static function settings_page() {
        if ( ! current_user_can('manage_options') ) return;

        $opts = get_option('dcw_mml_settings', []);
        $opts = wp_parse_args($opts, [
            'enabled'=>0,'title'=>'','message'=>'','logo'=>'',
            'bg_color'=>'#0f172a','bg_image'=>'','bg_size'=>'cover','bg_position'=>'center center',
            'title_color'=>'#ffffff','text_color'=>'#e5e7eb',
            'countdown_enable'=>0,'end_datetime'=>'',
            'preview_key'=>'','preview_allow_one'=>0,
        ]);

        $end_val = $opts['end_datetime'];
        if ( $end_val && strpos($end_val,'T') === false ) $end_val = str_replace(' ', 'T', $end_val);
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'DCW Maintenance Mode Lite', 'dcw-maintenance-mode-lite' ); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields('dcw_mml_group'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="enabled"><?php echo esc_html__( 'Enable maintenance mode', 'dcw-maintenance-mode-lite' ); ?></label>
                            <?php echo wp_kses_post( self::tip( __( 'Shows the maintenance page to visitors. Admins are bypassed. Popular caches are auto-purged when you toggle.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td><input type="checkbox" name="dcw_mml_settings[enabled]" id="enabled" value="1" <?php checked(1, $opts['enabled']); ?>></td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="title"><?php echo esc_html__( 'Title', 'dcw-maintenance-mode-lite' ); ?></label>
                            <?php echo wp_kses_post( self::tip( __( 'Main headline displayed on the page. Keep it short, e.g., “We’ll be back soon”.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td><input type="text" class="regular-text" name="dcw_mml_settings[title]" id="title" value="<?php echo esc_attr($opts['title']); ?>" placeholder="<?php echo esc_attr__( "We'll be back soon", 'dcw-maintenance-mode-lite' ); ?>"></td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="message"><?php echo esc_html__( 'Message (HTML allowed)', 'dcw-maintenance-mode-lite' ); ?></label>
                            <?php echo wp_kses_post( self::tip( __( 'Body text under the title. You can use simple HTML like links and paragraphs.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td><?php
                            wp_editor(
                                wp_kses_post($opts['message']),
                                'dcw_mml_message',
                                ['textarea_name'=>'dcw_mml_settings[message]', 'textarea_rows'=>6]
                            );
                        ?></td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php echo esc_html__( 'Logo', 'dcw-maintenance-mode-lite' ); ?>
                            <?php echo wp_kses_post( self::tip( __( 'Optional logo above the title. PNG/SVG recommended; around 180px width.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td>
                            <input type="text" name="dcw_mml_settings[logo]" id="dcw_mml_logo" value="<?php echo esc_attr($opts['logo']); ?>" class="regular-text" placeholder="<?php echo esc_attr__( 'Logo URL', 'dcw-maintenance-mode-lite' ); ?>">
                            <button type="button" class="button dcw-media-btn" data-target="#dcw_mml_logo"><?php echo esc_html__( 'Upload/Select', 'dcw-maintenance-mode-lite' ); ?></button>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="bg_color"><?php echo esc_html__( 'Background color', 'dcw-maintenance-mode-lite' ); ?></label>
                            <?php echo wp_kses_post( self::tip( __( 'Solid page background (and fallback behind the image).', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td>
                            <input type="text" name="dcw_mml_settings[bg_color]" id="bg_color" value="<?php echo esc_attr($opts['bg_color']); ?>" class="dcw-color-field" data-default-color="#0f172a">
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php echo esc_html__( 'Background image', 'dcw-maintenance-mode-lite' ); ?>
                            <?php echo wp_kses_post( self::tip( __( 'Optional hero image behind the content. Use high-contrast title/text colors over busy images.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td>
                            <input type="text" name="dcw_mml_settings[bg_image]" id="dcw_mml_bg_image" value="<?php echo esc_attr($opts['bg_image']); ?>" class="regular-text" placeholder="<?php echo esc_attr__( 'Image URL', 'dcw-maintenance-mode-lite' ); ?>">
                            <button type="button" class="button dcw-media-btn" data-target="#dcw_mml_bg_image"><?php echo esc_html__( 'Upload/Select', 'dcw-maintenance-mode-lite' ); ?></button>
                            <p>
                                <label>
                                    <?php echo esc_html__( 'Size', 'dcw-maintenance-mode-lite' ); ?>
                                    <?php echo wp_kses_post( self::tip( __( 'Cover fills the area (cropped). Contain fits entirely (may letterbox).', 'dcw-maintenance-mode-lite' ) ) ); ?>
                                    <select name="dcw_mml_settings[bg_size]">
                                        <option value="cover" <?php selected('cover', $opts['bg_size']); ?>><?php echo esc_html__( 'Cover (fill)', 'dcw-maintenance-mode-lite' ); ?></option>
                                        <option value="contain" <?php selected('contain', $opts['bg_size']); ?>><?php echo esc_html__( 'Contain (fit)', 'dcw-maintenance-mode-lite' ); ?></option>
                                    </select>
                                </label>
                                &nbsp;&nbsp;
                                <label>
                                    <?php echo esc_html__( 'Position', 'dcw-maintenance-mode-lite' ); ?>
                                    <?php echo wp_kses_post( self::tip( __( 'Anchor point for the background image (e.g., Center Center).', 'dcw-maintenance-mode-lite' ) ) ); ?>
                                    <select name="dcw_mml_settings[bg_position]">
                                        <?php foreach(['top left','top center','top right','center left','center center','center right','bottom left','bottom center','bottom right'] as $pos): ?>
                                            <option value="<?php echo esc_attr($pos); ?>" <?php selected($pos, $opts['bg_position']); ?>><?php echo esc_html(ucwords($pos)); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </label>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php echo esc_html__( 'Title color', 'dcw-maintenance-mode-lite' ); ?>
                            <?php echo wp_kses_post( self::tip( __( 'Color for the main H1 title. Ensure strong contrast with the background.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td><input type="text" name="dcw_mml_settings[title_color]" value="<?php echo esc_attr($opts['title_color']); ?>" class="dcw-color-field" data-default-color="#ffffff"></td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php echo esc_html__( 'Text color', 'dcw-maintenance-mode-lite' ); ?>
                            <?php echo wp_kses_post( self::tip( __( 'Color for the message and countdown text. Typically softer than the title.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td><input type="text" name="dcw_mml_settings[text_color]" value="<?php echo esc_attr($opts['text_color']); ?>" class="dcw-color-field" data-default-color="#e5e7eb"></td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php echo esc_html__( 'Countdown', 'dcw-maintenance-mode-lite' ); ?>
                            <?php echo wp_kses_post( self::tip( __( 'Show a live countdown. We also return HTTP 503 with Retry-After so search engines know this is temporary.', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td>
                            <label><input type="checkbox" name="dcw_mml_settings[countdown_enable]" value="1" <?php checked(1, $opts['countdown_enable']); ?>> <?php echo esc_html__( 'Enable', 'dcw-maintenance-mode-lite' ); ?></label><br>
                            <input type="datetime-local" name="dcw_mml_settings[end_datetime]" value="<?php echo esc_attr($end_val); ?>" class="regular-text">
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php echo esc_html__( 'Preview bypass', 'dcw-maintenance-mode-lite' ); ?>
                            <?php echo wp_kses_post( self::tip( __( 'Set a secret key so approved viewers can access the normal site using ?dcw_preview=YOUR_KEY (2-hour cookie).', 'dcw-maintenance-mode-lite' ) ) ); ?>
                        </th>
                        <td>
                            <input type="text" name="dcw_mml_settings[preview_key]" value="<?php echo esc_attr($opts['preview_key']); ?>" class="regular-text" placeholder="<?php echo esc_attr__( 'Optional key', 'dcw-maintenance-mode-lite' ); ?>">
                            <p class="description"><?php echo wp_kses_post( __( 'When set, visit <code>?dcw_preview=YOUR_KEY</code> to bypass for 2 hours.', 'dcw-maintenance-mode-lite' ) ); ?></p>
                            <label style="display:block;margin-top:6px;">
                                <input type="checkbox" name="dcw_mml_settings[preview_allow_one]" value="1" <?php checked(1, $opts['preview_allow_one']); ?>>
                                <?php echo wp_kses_post( __( 'Also allow <code>?dcw_preview=1</code>', 'dcw-maintenance-mode-lite' ) ); ?>
                                <?php echo wp_kses_post( self::tip( __( 'Enable a generic bypass link for convenience (use with care).', 'dcw-maintenance-mode-lite' ) ) ); ?>
                            </label>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>

                <?php
                    // Secure one-click Preview URL for admins (no 503; uses nonce).
                    $dcw_preview_url = add_query_arg(
                        array(
                            'dcw_mml_preview' => '1',
                            '_dcw_nonce'      => wp_create_nonce('dcw_mml_preview'),
                        ),
                        home_url('/')
                    );
                ?>
                <p style="margin-top:10px;">
                    <a href="<?php echo esc_url( $dcw_preview_url ); ?>" target="_blank" rel="noopener noreferrer" class="button button-secondary">
                        <?php echo esc_html__( 'Preview', 'dcw-maintenance-mode-lite' ); ?>
                    </a>
                    <span class="description" style="margin-left:8px;">
                        <?php echo esc_html__( 'Opens a new tab showing the maintenance page without enabling it (admins only).', 'dcw-maintenance-mode-lite' ); ?>
                    </span>
                </p>
            </form>
        </div>
        <?php
    }

    /** Add contextual Help tabs on this settings screen. */
    public static function add_help_tabs(){
        $screen = get_current_screen();
        if ( ! $screen || $screen->id !== self::$hook ) return;

        // Overview
        $screen->add_help_tab([
            'id'      => 'dcw_mml_overview',
            'title'   => esc_html__( 'Overview', 'dcw-maintenance-mode-lite' ),
            'content' => wp_kses_post(
                '<p><strong>DCW Maintenance Mode Lite</strong> shows a clean maintenance/coming-soon page to visitors while you work on your site. Admins are never blocked.</p>
                <ul>
                    <li>Logo, Title, Message</li>
                    <li>Background color or image (cover/contain, position)</li>
                    <li>Title/Text colors</li>
                    <li>Optional countdown (sends HTTP 503 + Retry-After)</li>
                    <li>Preview bypass via secret URL (2-hour cookie)</li>
                    <li>Auto-purge popular caches when toggled</li>
                </ul>'
            ),
        ]);

        // Settings
        $screen->add_help_tab([
            'id'      => 'dcw_mml_settings',
            'title'   => esc_html__( 'Settings', 'dcw-maintenance-mode-lite' ),
            'content' => wp_kses_post(
                '<p><strong>Key fields:</strong></p>
                <ul>
                    <li><em>Enable maintenance mode</em> — Show maintenance page to visitors.</li>
                    <li><em>Background image → Size</em> — Cover (fill), Contain (fit).</li>
                    <li><em>Background image → Position</em> — Anchor point (e.g., Center Center).</li>
                    <li><em>Title/Text colors</em> — Ensure strong contrast with your background.</li>
                    <li><em>Countdown</em> — Live timer to a date/time; also sets Retry-After.</li>
                </ul>'
            ),
        ]);

        // Preview
        $screen->add_help_tab([
            'id'      => 'dcw_mml_preview',
            'title'   => esc_html__( 'Preview Bypass', 'dcw-maintenance-mode-lite' ),
            'content' => wp_kses_post(
                '<p>Set a <strong>Preview key</strong> and share <code>?dcw_preview=YOUR_KEY</code>. After first visit, a 2-hour cookie bypasses maintenance.</p>
                <p>Optional: enable <code>?dcw_preview=1</code> for a generic link (use with care).</p>'
            ),
        ]);

        // Troubleshooting
        $screen->add_help_tab([
            'id'      => 'dcw_mml_troubleshoot',
            'title'   => esc_html__( 'Troubleshooting', 'dcw-maintenance-mode-lite' ),
            'content' => wp_kses_post(
                '<ul>
                    <li><strong>Still seeing normal site?</strong> You may have a bypass cookie. Try a private window.</li>
                    <li><strong>Old page stuck?</strong> Purge host/plugin/CDN caches. We auto-purge when toggled, but some CDNs may need manual purge.</li>
                    <li><strong>Locked out?</strong> Admins are never blocked; go to <code>/wp-login.php</code>.</li>
                </ul>'
            ),
        ]);

        // Optional sidebar.
        $screen->set_help_sidebar(
            wp_kses_post(
                '<p><strong>DCWireNet</strong><br>Maintenance Mode Lite</p>
                 <p>Keep titles short, contrast strong, and images centered.</p>'
            )
        );
    }

    /** Purge on toggle flip (OFF<->ON). */
    public static function maybe_purge_on_toggle( $option, $old, $new ){
        if ( $option !== 'dcw_mml_settings' || ! is_array($new) ) return;
        $old_on = isset($old['enabled']) ? (int) $old['enabled'] : 0;
        $new_on = isset($new['enabled']) ? (int) $new['enabled'] : 0;
        if ( $old_on !== $new_on ) self::purge_all_caches();
    }

    /** Call popular cache purges if present (safe no-ops otherwise). */
    public static function purge_all_caches(){
        if ( function_exists('wp_cache_flush') ) { @wp_cache_flush(); }
        if ( function_exists('rocket_clean_domain') ) { rocket_clean_domain(); }
        if ( function_exists('rocket_clean_minify') ) { rocket_clean_minify(); }
        if ( class_exists('\\LiteSpeed\\Purge') && method_exists('\\LiteSpeed\\Purge','purge_all') ) { \LiteSpeed\Purge::purge_all(); }
        if ( function_exists('wp_cache_clear_cache') ) { wp_cache_clear_cache(); }
        if ( function_exists('w3tc_flush_all') ) { w3tc_flush_all(); }
        if ( function_exists('sg_cachepress_purge_cache') ) { sg_cachepress_purge_cache(); }
        if ( class_exists('Swift_Performance_Cache') ) { Swift_Performance_Cache::clear_all_cache(); }
        if ( class_exists('Breeze_PurgeCache') ) { Breeze_PurgeCache::breeze_cache_flush(); }
        if ( function_exists('cachify_flush_cache') ) { cachify_flush_cache(); }
        if ( class_exists('autoptimizeCache') && method_exists('autoptimizeCache','clearall') ) { try { autoptimizeCache::clearall(); } catch ( \Throwable $e ) {} }
    }
}
