<?php
if ( ! defined('ABSPATH') ) exit;

class DCW_MML_Frontend {

    public static function init(){
        add_action( 'template_redirect', [__CLASS__, 'maybe_serve_maintenance'], 0 );
        add_action( 'wp_enqueue_scripts', [__CLASS__, 'enqueue'] );
    }

    /** Get merged options with sane defaults. */
    private static function opts(){
        $o = get_option('dcw_mml_settings', []);
        return wp_parse_args($o, [
            'enabled'           => 0,
            'title'             => "We'll be back soon",
            'message'           => '<p>Our site is undergoing scheduled maintenance. Thank you for your patience.</p>',
            'logo'              => '',
            'bg_color'          => '#0f172a',
            'bg_image'          => '',
            'bg_size'           => 'cover',
            'bg_position'       => 'center center',
            'title_color'       => '#ffffff',
            'text_color'        => '#e5e7eb',
            'countdown_enable'  => 0,
            'end_datetime'      => '',
            // existing bypass (secret key -> cookie)
            'preview_key'       => '',
            'preview_allow_one' => 0,
        ]);
    }

    /** Admin-only preview flag: ?dcw_mml_preview=1&_dcw_nonce=... */
    private static function is_preview_request(){
        if ( ! is_user_logged_in() || ! current_user_can('manage_options') ) return false;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $preview = isset($_GET['dcw_mml_preview']) ? sanitize_text_field( wp_unslash( $_GET['dcw_mml_preview'] ) ) : '';
        if ( $preview !== '1' ) return false;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $nonce = isset($_GET['_dcw_nonce']) ? sanitize_text_field( wp_unslash( $_GET['_dcw_nonce'] ) ) : '';
        if ( ! $nonce || ! wp_verify_nonce( $nonce, 'dcw_mml_preview' ) ) return false;
        return true;
    }

    /** Existing secret bypass (?dcw_preview=KEY sets a cookie for 2h). */
    private static function has_secret_bypass_cookie(){
        $opts = self::opts();
        $cookie_name = 'dcw_mml_preview';
        $preview_key = isset($opts['preview_key']) ? trim($opts['preview_key']) : '';
        $allow_one   = ! empty($opts['preview_allow_one']);

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset($_GET['dcw_preview']) ) {
            // Read-only behavior change; param sanitized.
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $val = sanitize_text_field( wp_unslash( $_GET['dcw_preview'] ) );
            $match = false;
            if ( $preview_key !== '' && hash_equals($preview_key, $val) ) $match = true;
            if ( $allow_one && $val === '1' ) $match = true;

            if ( $match ) {
                @setcookie(
                    $cookie_name,
                    wp_hash('dcw_' . $val),
                    time() + 2 * HOUR_IN_SECONDS,
                    COOKIEPATH ? COOKIEPATH : '/',
                    COOKIE_DOMAIN,
                    is_ssl(),
                    true
                );
                // Set superglobal so this very request bypasses too.
                $_COOKIE[$cookie_name] = wp_hash('dcw_' . $val);
            }
        }

        return isset($_COOKIE[$cookie_name]) && is_string($_COOKIE[$cookie_name]);
    }

    /** Should maintenance render for this request? (includes admin preview) */
    private static function is_active_for_request(){
        // If admin explicitly previewing, force ON
        if ( self::is_preview_request() ) return true;

        $opts = self::opts();
        if ( empty($opts['enabled']) ) return false;

        // Admins bypass when not explicitly previewing
        if ( current_user_can('manage_options') ) return false;

        // Secret preview cookie bypass
        if ( self::has_secret_bypass_cookie() ) return false;

        return true;
    }

    public static function enqueue(){
        // Enqueue when page will render (either real maintenance or admin preview)
        if ( ! self::is_active_for_request() ) return;
        wp_enqueue_style( 'dcw-mml-style', DCW_MML_URL . 'assets/css/style.css', [], DCW_MML_VERSION );
        wp_enqueue_script( 'dcw-mml-countdown', DCW_MML_URL . 'assets/js/countdown.js', [], DCW_MML_VERSION, true );
    }

    public static function maybe_serve_maintenance(){
        // Allow login, AJAX, REST to work normally
        if ( defined('DOING_AJAX') && DOING_AJAX ) return;

        $uri = '';
        if ( isset($_SERVER['REQUEST_URI']) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $uri = sanitize_text_field( wp_unslash( (string) $_SERVER['REQUEST_URI'] ) );
        }
        if ( strpos($uri, 'wp-login.php') !== false ) return;
        if ( strpos($uri, '/wp-json') === 0 ) return;

        $is_preview = self::is_preview_request();

        if ( ! self::is_active_for_request() ) return;

        $opts = self::opts();

        // Headers: 503 for real maintenance, 200 for preview (noindex)
        if ( $is_preview ) {
            status_header(200);
            header('X-Robots-Tag: noindex, nofollow', true);
            nocache_headers();
        } else {
            $retry_after = 3600;
            if ( ! empty($opts['end_datetime']) ) {
                $normalized = str_replace('T', ' ', $opts['end_datetime']);
                $ts = strtotime($normalized);
                if ( $ts && $ts > time() ) $retry_after = max(60, $ts - time());
            }
            status_header(503);
            header('Retry-After: ' . (int) $retry_after);
            nocache_headers();
            send_nosniff_header();
        }

        // Build data for template
        $data = [
            'title'    => !empty($opts['title']) ? $opts['title'] : "We'll be back soon",
            'message'  => !empty($opts['message']) ? $opts['message'] : '<p>Our site is undergoing scheduled maintenance. Thank you for your patience.</p>',
            'logo'     => isset($opts['logo']) ? esc_url($opts['logo']) : '',
            'bg'       => isset($opts['bg_color']) ? $opts['bg_color'] : '#0f172a',
            'bg_img'   => isset($opts['bg_image']) ? esc_url($opts['bg_image']) : '',
            'bg_size'  => (isset($opts['bg_size']) && in_array($opts['bg_size'], ['cover','contain'], true)) ? $opts['bg_size'] : 'cover',
            'bg_pos'   => isset($opts['bg_position']) ? $opts['bg_position'] : 'center center',
            'title_c'  => isset($opts['title_color']) ? $opts['title_color'] : '#ffffff',
            'text_c'   => isset($opts['text_color'])  ? $opts['text_color']  : '#e5e7eb',
            'cd_en'    => !empty($opts['countdown_enable']),
            'end'      => isset($opts['end_datetime']) ? $opts['end_datetime'] : '',
        ];

        self::render_page($data);
        exit;
    }

    private static function render_page( $d ){
        $inline = '--dcw-bg:' . esc_attr($d['bg']) . '; --dcw-title:' . esc_attr($d['title_c']) . '; --dcw-text:' . esc_attr($d['text_c']) . ';';
        if ( ! empty($d['bg_img']) ){
            $inline .= ' background-image:url(' . esc_url($d['bg_img']) . '); background-size:' . esc_attr($d['bg_size']) . '; background-position:' . esc_attr($d['bg_pos']) . '; background-repeat:no-repeat;';
        }
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title><?php echo esc_html($d['title']); ?></title>
            <?php wp_head(); ?>
        </head>
        <body class="dcw-mml" style="<?php echo esc_attr( $inline ); ?>">
            <main class="dcw-wrap">
                <?php if ( $d['logo'] ): ?>
                    <img src="<?php echo esc_url($d['logo']); ?>" alt="<?php echo esc_attr__( 'Logo', 'dcw-maintenance-mode-lite' ); ?>" class="dcw-logo">
                <?php endif; ?>
                <h1 class="dcw-title"><?php echo esc_html($d['title']); ?></h1>
                <div class="dcw-msg"><?php echo wp_kses_post($d['message']); ?></div>

                <?php if ( $d['cd_en'] && ! empty($d['end']) ): ?>
                    <div id="dcw-countdown" data-end="<?php echo esc_attr($d['end']); ?>" class="dcw-countdown"></div>
                <?php endif; ?>
            </main>
            <?php wp_footer(); ?>
        </body>
        </html>
        <?php
    }
}
